﻿<?php
// =========================================================================
// ficha_avaliacao_detalhada.php - Ecrã de Entrada de Dados (Gestor/Avaliador)
// Implementa a pontuação automática, AGORA EDITÁVEL PELO AVALIADOR.
// Inclui navegação fácil entre os trimestres.
// =========================================================================

// --- CONFIGURAÇÃO DE CONEXÃO ---
define('DB_HOST', 'localhost');
define('DB_PORT', '3306');
define('DB_NAME', 'sismag'); 
define('DB_USER', 'root');
define('DB_PASS', '');

$pdo = null;

try {
    $dsn = "mysql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME . ";charset=utf8";
    $pdo = new PDO($dsn, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Se a conexão falhar, interrompe o script
    die("ERRO DE CONEXÃO COM A BASE DE DADOS 'sismag': " . $e->getMessage());
}

// --- DEFINIÇÕES DE DATA E TRIMESTRE ---
$ANO_ATUAL = date('Y'); 
$TRIMESTRES_DISPONIVEIS = [
    1 => 'I Trimestre',
    2 => 'II Trimestre',
    3 => 'III Trimestre',
    4 => 'IV Trimestre',
];

// --- VARIÁVEIS DE REQUISIÇÃO (CORRIGIDO: Definidas antes do cálculo das datas) ---
$id_selecionado = isset($_REQUEST['id']) ? (int)$_REQUEST['id'] : null;
// Garante que o trimestre seja 1 se não estiver definido
$trimestre_url = isset($_REQUEST['trimestre']) ? (int)$_REQUEST['trimestre'] : 1; 

// --- FUNÇÃO PARA CALCULAR DATAS DO TRIMESTRE ---
/**
 * Retorna as datas de início e fim do trimestre civil.
 * Estas datas são usadas para filtrar presenças.
 */
function get_trimestre_dates($ano, $trimestre) {
    // Ajuste estas datas se o seu calendário escolar for diferente.
    switch ($trimestre) {
        case 1:
            return ['inicio' => "$ano-01-01", 'fim' => "$ano-03-31"];
        case 2:
            return ['inicio' => "$ano-04-01", 'fim' => "$ano-06-30"];
        case 3:
            return ['inicio' => "$ano-07-01", 'fim' => "$ano-09-30"];
        case 4:
            return ['inicio' => "$ano-10-01", 'fim' => "$ano-12-31"];
        default:
            // Fallback para o ano inteiro, se o trimestre for inválido
            return ['inicio' => "$ano-01-01", 'fim' => "$ano-12-31"];
    }
}
// CHAMADA AGORA CORRETA, POIS $trimestre_url FOI DEFINIDA
$datas_trimestre = get_trimestre_dates($ANO_ATUAL, $trimestre_url);


// --- VARIÁVEIS DE ESTADO E INICIALIZAÇÃO ---
$usuario_selecionado = null;
$dados_avaliacao_salvos = [];
$mensagem = '';
$ponto_sugerido_item_I = 5; 
$ponto_sugerido_item_II = 5;
$ponto_sugerido_item_III = 5; 
$ponto_sugerido_item_IV = 5; 


if (!$id_selecionado || !isset($TRIMESTRES_DISPONIVEIS[$trimestre_url])) {
    die("<h1>Erro</h1><p>ID do funcionário inválido ou Trimestre não reconhecido.</p>");
}

// =================================================================
// 1. CÁLCULO DA PONTUAÇÃO AUTOMÁTICA (ITEM I: Planificação)
// =================================================================
$stmt_planos = $pdo->prepare("
    SELECT COUNT(id) as num_planos 
    FROM plano_aula_mensal 
    WHERE usuario_id = ? AND ano = ? AND trimestre = ?
");
$stmt_planos->execute([$id_selecionado, $ANO_ATUAL, $trimestre_url]);
$resultado_planos = $stmt_planos->fetch();
$num_planos_enviados = $resultado_planos['num_planos'];

if ($num_planos_enviados == 0) {
    $ponto_sugerido_item_I = 5;
} elseif ($num_planos_enviados == 1) {
    $ponto_sugerido_item_I = 10;
} elseif ($num_planos_enviados == 2) {
    $ponto_sugerido_item_I = 15;
} else { // 3 ou mais
    $ponto_sugerido_item_I = 20;
}


// --- BLOCO DE DADOS DE LIGAÇÃO (DISCIPLINAS) ---
$stmt_disciplinas = $pdo->prepare("
    SELECT disciplina_id FROM atribuicoes
    WHERE professor_id = ? AND ano_lectivo = ?
    GROUP BY disciplina_id
");
$stmt_disciplinas->execute([$id_selecionado, $ANO_ATUAL]);
$disciplina_ids = $stmt_disciplinas->fetchAll(PDO::FETCH_COLUMN);

$disciplinas_atribuidas = !empty($disciplina_ids);
$placeholders = $disciplinas_atribuidas ? implode(',', array_fill(0, count($disciplina_ids), '?')) : '0';
$params_notas = $disciplinas_atribuidas ? array_merge($disciplina_ids, [$ANO_ATUAL, $trimestre_url]) : [$ANO_ATUAL, $trimestre_url];
// --- FIM DO BLOCO DE DADOS DE LIGAÇÃO ---


// =================================================================
// 2. CÁLCULO DA PONTUAÇÃO AUTOMÁTICA (ITEM II: Progresso do Aluno)
// =================================================================
if ($trimestre_url == 1) {
    $coluna_nota = 'mt1';
} elseif ($trimestre_url == 2) {
    $coluna_nota = 'mt2'; 
} elseif ($trimestre_url == 3) {
    $coluna_nota = 'mt3'; 
} else {
    $coluna_nota = 'mf'; 
}

$media_alunos = 0;

if ($disciplinas_atribuidas) {
    $stmt_media = $pdo->prepare("
        SELECT AVG({$coluna_nota}) as media_geral  
        FROM notas  
        WHERE disciplina_id IN ({$placeholders}) AND ano = ? AND trimestre = ?
    ");
    $stmt_media->execute($params_notas);
    $resultado_media = $stmt_media->fetch();
    $media_alunos = $resultado_media['media_geral'] ?? 0;
}

if ($media_alunos >= 17.0) {
    $ponto_sugerido_item_II = 20;
} elseif ($media_alunos >= 14.0) {
    $ponto_sugerido_item_II = 15;
} elseif ($media_alunos >= 10.0) {
    $ponto_sugerido_item_II = 10;
} else {
    $ponto_sugerido_item_II = 5;
}
$media_alunos_formatada = number_format($media_alunos, 2);


// =================================================================
// 3. CÁLCULO DA PONTUAÇÃO AUTOMÁTICA (ITEM III: Responsabilidade - Prazo de Lançamento)
// =================================================================

// 3.1. Busca o prazo limite
$stmt_prazo = $pdo->prepare("
    SELECT data_limite FROM prazo_lancamento_notas 
    WHERE ano = ? AND trimestre = ?
");
$stmt_prazo->execute([$ANO_ATUAL, $trimestre_url]);
$prazo_limite = $stmt_prazo->fetchColumn();

// 3.2. Busca a data do último lançamento de nota do professor
$data_max_lancamento = null;

if ($disciplinas_atribuidas) {
    $stmt_max_data = $pdo->prepare("
        SELECT MAX(DATE(data_registo)) as data_max_lancamento  
        FROM notas  
        WHERE disciplina_id IN ({$placeholders}) AND ano = ? AND trimestre = ?
    ");
    $stmt_max_data->execute($params_notas);
    $data_max_lancamento = $stmt_max_data->fetchColumn(); 
}

$status_prazo = 'Prazo não definido ou notas não lançadas.';

if ($prazo_limite && $data_max_lancamento) {
    $data1 = new DateTime($data_max_lancamento); // Data de submissão (último lançamento)
    $data2 = new DateTime($prazo_limite);        // Data limite
    
    $dias_atraso = ceil(($data1->getTimestamp() - $data2->getTimestamp()) / (60 * 60 * 24)); 

    if ($dias_atraso < 0) {
        $ponto_sugerido_item_III = 20;
        $status_prazo = "Concluído com " . abs($dias_atraso) . " dias de antecedência.";
    } elseif ($dias_atraso == 0) {
        $ponto_sugerido_item_III = 15;
        $status_prazo = "Concluído no prazo (limite: " . date('d/m/Y', strtotime($prazo_limite)) . ").";
    } elseif ($dias_atraso >= 1 && $dias_atraso <= 3) {
        $ponto_sugerido_item_III = 10;
        $status_prazo = "Concluído com $dias_atraso dias de atraso.";
    } else { // $dias_atraso >= 4
        $ponto_sugerido_item_III = 5;
        $status_prazo = "Concluído com $dias_atraso dias de atraso (mais de 3 dias).";
    }

} else {
    $ponto_sugerido_item_III = 5;
    if ($prazo_limite && !$data_max_lancamento) {
        $status_prazo = "Prazo (" . date('d/m/Y', strtotime($prazo_limite)) . ") ultrapassado. Notas não lançadas.";
    } elseif (!$prazo_limite) {
        $status_prazo = "Prazo Limite não definido pela Subdireção Pedagógica.";
    }
}


// =================================================================
// 4. CÁLCULO DA PONTUAÇÃO AUTOMÁTICA (ITEM IV: Aperfeiçoamento Profissional)
// =================================================================
$stmt_presencas_trimestre = $pdo->prepare("
    SELECT COUNT(id) as total_presencas  
    FROM presencas_seminario 
    WHERE professor_id = ?  
    AND professor_id != 0  
    -- FILTRO: Usa as datas do trimestre civil
    AND data_presenca BETWEEN ? AND ? 
");
// Parâmetros: [ID do Professor, Data de Início do Trimestre, Data de Fim do Trimestre]
$stmt_presencas_trimestre->execute([$id_selecionado, $datas_trimestre['inicio'], $datas_trimestre['fim']]);
$resultado_presencas = $stmt_presencas_trimestre->fetch();
$dias_presenca = $resultado_presencas['total_presencas'];

if ($dias_presenca >= 10) {
    $ponto_sugerido_item_IV = 20;
} elseif ($dias_presenca >= 7) {
    $ponto_sugerido_item_IV = 15;
} elseif ($dias_presenca >= 4) {
    $ponto_sugerido_item_IV = 10;
} else {
    $ponto_sugerido_item_IV = 5;
}


// --- DEFINIÇÕES DA AVALIAÇÃO COM PESOS E CRITÉRIOS ---
$detalhes_avaliacao = [
    'I. Qualidade do Ensino e Aprendizagem (Peso: 0,2)' => [
        'I_QA' => [
            'nome' => 'Qualidade do Processo de Ensino e Aprendizagem (Planificação)', 
            'max' => 20,
            'peso' => 0.2,
            'pontuacao_sugerida' => $ponto_sugerido_item_I, // AUTOMÁTICO - ITEM I
            'criterios' => [
                5 => 'Não planifica os conteúdos; Não se empenha no cumprimento da matéria; ministrou apenas 25% das aulas.',
                10 => 'Planifica algumas vezes; Empenha-se pouco no cumprimento do programado.',
                15 => 'Planifica com regularidade; Empenha-se no cumprimento da matéria programada; Ministrou até 75% da matéria.',
                20 => 'Planifica sempre os conteúdos; Empenha-se bastante no cumprimento da matéria programada; Ministrou até 100% da matéria.',
            ]
        ],
    ],
    'II. Progresso do Aluno (Peso: 0,3)' => [
        'II_PA' => [
            'nome' => 'Progresso do Aluno ou Desenvolvimento do Aluno (Estatística de Notas)', 
            'max' => 20,
            'peso' => 0.3,
            'pontuacao_sugerida' => $ponto_sugerido_item_II, // AUTOMÁTICO - ITEM II
            'criterios' => [
                5 => 'O trabalho do professor não resulta no crescimento aceitável (até 49%); O nível de aprendizagem é muito fraco.',
                10 => 'O trabalho resulta no crescimento lento dos alunos; O nível de aprendizagem dos alunos é regular (50%-74%).',
                15 => 'O trabalho do professor resulta no progresso aceitável e mensurável do desenvolvimento do aluno; O nível de aprendizagem dos alunos é bom (75%-89%).',
                20 => 'O trabalho do professor resulta em alto nível de realizações com todos os alunos; O nível de aprendizagem dos alunos é muito bom (90%-100%).',
            ]
        ],
    ],
    'III. Responsabilidade (Peso: 0,1)' => [
        'III_R' => [
            'nome' => 'Responsabilidade (Lançamento de Notas no Prazo)', 
            'max' => 20,
            'peso' => 0.1,
            'pontuacao_sugerida' => $ponto_sugerido_item_III, // AUTOMÁTICO - ITEM III
            'criterios' => [
                5 => 'É normalmente pouco cumpridor das normas; atrasa com frequência e comete muitas faltas; Dificilmente entrega o expediente.',
                10 => 'É normalmente pouco disciplinado e responsável; Atrasa algumas vezes e comete algumas faltas; Entrega o expediente quando pressionado.',
                15 => 'É disciplinado e assume as responsabilidades; Dificilmente atrasa e comete poucas faltas; Entrega o expediente dentro do prazo.',
                20 => 'É muito disciplinado, assume plenamente as suas responsabilidades e manifesta elevado sentido patriótico; Não atrasa e não comete faltas; Entrega o expediente antes do prazo.',
            ]
        ],
    ],
    'IV. Aperfeiçoamento Profissional e Inovação Pedagógica (Peso: 0,2)' => [
        'IV_AP' => [
            'nome' => 'Aperfeiçoamento Profissional (Participação em Seminários)', 
            'max' => 20,
            'peso' => 0.2,
            'pontuacao_sugerida' => $ponto_sugerido_item_IV, // AUTOMÁTICO - ITEM IV
            'criterios' => [
                5 => 'Mostra pouco interesse em adquirir novos conhecimentos; Não participa em acções de formação e revela resistência à mudança.',
                10 => 'Mostra algum interesse em aumentar os seus conhecimentos e aperfeiçoar o seu trabalho, hesita perante soluções menos frequentes.',
                15 => 'Revela interesse em aumentar os seus conhecimentos e em aperfeiçoar o seu trabalho; Adapta-se às novas exigências e a situações pouco frequentes.',
                20 => 'Revela interesse metódico e sistemático em melhorar os conhecimentos; Sua adaptação à mudança é excelente, destacando-se em desenvolver e criar novos métodos.',
            ]
        ],
    ],
    'V. Relações Humanas (Peso: 0,2)' => [
        'V_RH' => [
            'nome' => 'Relações Humanas', 
            'max' => 20,
            'peso' => 0.2,
            'pontuacao_sugerida' => null, // Manual
            'criterios' => [
                5 => 'Provoca atritos frequentes e pouco contribui para bom ambiente; Tem dificuldade de se relacionar com colegas/alunos/encarregados e é passivo no trabalho de grupo.',
                10 => 'Pouco contribui para o bom ambiente; Tem dificuldade de se relacionar; Tem dificuldade de se integrar sendo quase sempre passivo no trabalho de grupo.',
                15 => 'Contribui sempre para um bom ambiente; Estabelece relações cordiais; Integra-se no grupo com espírito de cooperação, quando solicitado.',
                20 => 'Sua maneira de estar incentiva sempre um bom ambiente de trabalho; Integra-se com muita facilidade no grupo, intervindo com eficiência.',
            ]
        ],
    ],
];
$pontuacoes_validas = [5, 10, 15, 20];

// --- LÓGICA DE BUSCA DE DADOS E PRÉ-PREENCHIMENTO ---

// 1. Busca dados do funcionário
$stmt = $pdo->prepare("SELECT id, nome FROM usuarios WHERE id = ?");
$stmt->execute([$id_selecionado]);
$usuario_selecionado = $stmt->fetch();

if (!$usuario_selecionado) {
    die("<h1>Erro</h1><p>Funcionário não encontrado.</p>");
}

// 2. Busca dados de avaliação previamente salvos
$stmt_salvos = $pdo->prepare("SELECT item_codigo, pontuacao_atribuida FROM avaliacao_item WHERE usuario_id = ? AND ano = ? AND trimestre = ?");
$stmt_salvos->execute([$id_selecionado, $ANO_ATUAL, $trimestre_url]);
$itens_salvos = $stmt_salvos->fetchAll(PDO::FETCH_KEY_PAIR);

// 3. Monta o array de avaliação, usando o valor SALVO (se existir) ou o valor SUGERIDO
$total_ponderado_calculado = 0;
foreach ($detalhes_avaliacao as $seccao_titulo => $itens_seccao) {
    foreach ($itens_seccao as $codigo => $detalhe) {
        
        $is_automatico = isset($detalhe['pontuacao_sugerida']); 
        $ponto_sugerido = $detalhe['pontuacao_sugerida'] ?? 0;
        $ponto_atribuido = 0;

        if (isset($itens_salvos[$codigo])) {
            $ponto_atribuido = $itens_salvos[$codigo];
        } elseif ($is_automatico) {
            $ponto_atribuido = $ponto_sugerido;
        } else {
             $ponto_atribuido = 0;
        }

        $pontuacao_ponderada = $ponto_atribuido * $detalhe['peso'];
        $total_ponderado_calculado += $pontuacao_ponderada;

        $dados_avaliacao_salvos[$codigo] = [
            'ponto_atribuido' => $ponto_atribuido,
            'pontuacao_sugerida' => $ponto_sugerido, 
            'pontuacao_ponderada' => $pontuacao_ponderada,
            'peso' => $detalhe['peso'],
            'nome' => $detalhe['nome'],
            'automatico' => $is_automatico,
            'criterio_descricao' => $detalhe['criterios'][$ponto_atribuido] ?? 'Pontuação não atribuída ou inválida.',
        ];
    }
}


// --- LÓGICA DE SALVAR A AVALIAÇÃO (Completa) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['salvar_avaliacao'])) {
    
    $apreciacao = $_POST['apreciacao_avaliador'];
    $pontos_inseridos = $_POST['pontuacao_atribuida']; 
    $total_final_ponderado = 0;

    try {
        $pdo->beginTransaction();
        
        // 1. Apaga avaliações anteriores
        $stmt_delete = $pdo->prepare("DELETE FROM avaliacao_item WHERE usuario_id = ? AND ano = ? AND trimestre = ?");
        $stmt_delete->execute([$id_selecionado, $ANO_ATUAL, $trimestre_url]);

        // 2. Insere os pontos de cada item
        foreach ($detalhes_avaliacao as $seccao_titulo => $itens_seccao) {
            foreach ($itens_seccao as $codigo => $detalhe) {
                
                $ponto = (int)($pontos_inseridos[$codigo] ?? 0);
                
                if (!in_array($ponto, $pontuacoes_validas)) $ponto = 0;

                $pontuacao_ponderada = $ponto * $detalhe['peso'];
                $total_final_ponderado += $pontuacao_ponderada;
                
                if ($ponto > 0) {
                    $stmt_insert = $pdo->prepare("
                        INSERT INTO avaliacao_item (usuario_id, ano, trimestre, item_codigo, pontuacao_atribuida, data_avaliacao)
                        VALUES (?, ?, ?, ?, ?, NOW())
                    ");
                    $stmt_insert->execute([$id_selecionado, $ANO_ATUAL, $trimestre_url, $codigo, $ponto]);
                }
            }
        }
        
        // 3. Calcula a classificação qualitativa
        if ($total_final_ponderado >= 18) {
            $classificacao_qualitativa = 'Excelente';
        } elseif ($total_final_ponderado >= 15) {
            $classificacao_qualitativa = 'Muito Bom';
        } elseif ($total_final_ponderado >= 10) {
            $classificacao_qualitativa = 'Bom';
        } else {
            $classificacao_qualitativa = 'Insuficiente';
        }

        // 4. Salva a avaliação final
        $stmt_final = $pdo->prepare("
            INSERT INTO avaliacao_final (usuario_id, ano, trimestre, pontuacao_quantitativa, classificacao_qualitativa, apreciacao_avaliador)
            VALUES (?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
            pontuacao_quantitativa = VALUES(pontuacao_quantitativa),
            classificacao_qualitativa = VALUES(classificacao_qualitativa),
            apreciacao_avaliador = VALUES(apreciacao_avaliador)
        ");
        $stmt_final->execute([$id_selecionado, $ANO_ATUAL, $trimestre_url, $total_final_ponderado, $classificacao_qualitativa, $apreciacao]);

        $pdo->commit();

        // Recarrega os dados para exibir os valores salvos/finais
        header("Location: ficha_avaliacao_detalhada.php?id={$id_selecionado}&trimestre={$trimestre_url}&msg=sucesso");
        exit;

    } catch (PDOException $e) {
        $pdo->rollBack();
        $mensagem = "<p style='color:red;'>Erro ao salvar: " . $e->getMessage() . "</p>";
    }
}

// Atualiza o total final ponderado após o POST (ou mantém o cálculo inicial)
if (isset($_GET['msg']) && $_GET['msg'] === 'sucesso') {
    $stmt_final = $pdo->prepare("SELECT pontuacao_quantitativa FROM avaliacao_final WHERE usuario_id = ? AND ano = ? AND trimestre = ?");
    $stmt_final->execute([$id_selecionado, $ANO_ATUAL, $trimestre_url]);
    $resultado_final = $stmt_final->fetch();
    $total_final_ponderado = $resultado_final['pontuacao_quantitativa'] ?? $total_ponderado_calculado;
    $mensagem = "<p style='color:green; font-weight:bold;'>Avaliação salva com sucesso! Pontuação Final: " . number_format($total_final_ponderado, 2) . "</p>";
} else {
    $total_final_ponderado = $total_ponderado_calculado;
}

?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <title>Avaliar <?= htmlspecialchars($usuario_selecionado['nome']) ?> (<?= htmlspecialchars($TRIMESTRES_DISPONIVEIS[$trimestre_url]) ?>)</title>
    <style>
        /* ... Estilos CSS ... */
        body { font-family: Arial, sans-serif; background-color: #f4f4f4; margin: 20px; }
        .container { max-width: 1200px; margin: 0 auto; background-color: #fff; padding: 25px; border-radius: 8px; box-shadow: 0 0 15px rgba(0,0,0,0.1); }
        h1 { text-align: center; color: #007bff; border-bottom: 3px solid #007bff; padding-bottom: 10px; margin-bottom: 25px; }
        .info-header { margin-bottom: 20px; padding: 10px; background-color: #e9ecef; border-left: 5px solid #007bff; }
        .evaluation-table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        .evaluation-table th, .evaluation-table td { border: 1px solid #ddd; padding: 10px; text-align: left; vertical-align: top; }
        .evaluation-table th { background-color: #007bff; color: white; text-align: center; }
        .item-row.automatico td { background-color: #fff3e6; } 
        .form-select { padding: 8px; border: 1px solid #ccc; border-radius: 4px; width: 100px; }
        textarea { width: 100%; height: 100px; padding: 10px; border: 1px solid #ccc; border-radius: 4px; box-sizing: border-box; resize: vertical; }
        .result-box { margin-top: 20px; padding: 15px; background-color: #eaf7ed; border: 1px solid #c3e6cb; text-align: center; font-size: 1.1em; font-weight: bold; }
        .button-group { margin-top: 20px; text-align: right; }
        .button-group button { background-color: #28a745; color: white; padding: 10px 20px; border: none; border-radius: 4px; cursor: pointer; font-size: 1em; }
        .link-ficha { margin-top: 10px; display: inline-block; color: #007bff; text-decoration: none; font-size: 0.9em; margin-right: 20px; }
        .criterio-text { font-style: italic; color: #555; font-size: 0.9em; }
        
        .criterios-detalhados { 
            margin-top: 25px; 
            padding: 20px; 
            border: 1px solid #007bff; 
            background-color: #f8f9fa; 
            border-radius: 8px;
            display: none; 
        }
        .criterios-detalhados h3 {
            color: #007bff;
            border-bottom: 2px solid #ccc;
            padding-bottom: 5px;
            margin-top: 15px;
        }
        .criterios-detalhados th, .criterios-detalhados td {
            border: 1px solid #ccc;
            padding: 8px;
            vertical-align: top;
            font-size: 0.9em;
        }
        .criterios-detalhados th {
            background-color: #6c757d;
            color: white;
            width: 10%;
        }
        .show-criterios-btn {
            background-color: #ffc107;
            color: #212529;
            padding: 8px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 0.9em;
            margin-top: 10px;
        }
        .sugestao-auto {
            display: block;
            font-size: 0.75em;
            color: #d9534f;
            font-weight: bold;
        }
        /* Estilo para navegação entre trimestres */
        .trimestre-nav {
            text-align: center; 
            margin-bottom: 20px; 
            font-size: 1.1em; 
            padding: 10px; 
            background-color: #f0f8ff; 
            border: 1px solid #cce5ff;
            border-radius: 4px;
        }
        .trimestre-nav a {
            padding: 5px 10px;
            margin: 0 5px;
            border-radius: 4px;
            transition: background-color 0.3s;
        }
        .trimestre-nav a:hover {
            background-color: #e2e8f0;
        }
        .trimestre-nav .active {
            font-weight: bold; 
            color: white !important; 
            text-decoration: none; 
            background-color: #007bff;
            border: 1px solid #0056b3;
        }
        /* Fim dos Estilos CSS */
    </style>
</head>
<body>
    <div class="container">
        <h1>Avaliação de Desempenho Trimestral</h1>

        <div class="trimestre-nav">
            <span>Acessar Avaliação do: </span>
            <?php foreach ($TRIMESTRES_DISPONIVEIS as $num => $nome): ?>
                <?php 
                    $url_trimestre = "ficha_avaliacao_detalhada.php?id={$id_selecionado}&trimestre={$num}";
                    $classe_ativa = ($num == $trimestre_url) ? 'active' : '';
                ?>
                <a href="<?= $url_trimestre ?>" class="<?= $classe_ativa ?>">
                    <?= $nome ?>
                </a>
            <?php endforeach; ?>
        </div>
        <div class="info-header">
            <p><strong>Funcionário:</strong> <?= htmlspecialchars($usuario_selecionado['nome']) ?></p>
            <p><strong>Período em Avaliação:</strong> <span style="font-size: 1.2em; color: #dc3545; font-weight: bold;"><?= htmlspecialchars($TRIMESTRES_DISPONIVEIS[$trimestre_url]) . ' / ' . $ANO_ATUAL ?></span></p>
            <p style="color: darkorange; font-weight: bold;">
                **Nota:** Itens I, II, III e IV têm uma pontuação sugerida, mas podem ser **editados** pelo avaliador.
            </p>
            <p>
                **Item I (Planificação):** <?= $num_planos_enviados ?> planos enviados.
            </p>
            <p>
                **Item II (Progresso do Aluno):** Média Geral de Notas dos Alunos: <?= $media_alunos_formatada ?> valores (Usando a coluna **<?= strtoupper($coluna_nota) ?>**).
            </p>
            <p>
                **Item III (Responsabilidade):** Situação de Lançamento de Notas: <span style="font-weight: bold; color: #007bff;"><?= htmlspecialchars($status_prazo) ?></span>
            </p>
            <p>
                **Item IV (Aperfeiçoamento Profissional):** Total de Presenças em Seminários no Trimestre (<?= date('d/m/Y', strtotime($datas_trimestre['inicio'])) ?> a <?= date('d/m/Y', strtotime($datas_trimestre['fim'])) ?>): <span style="font-weight: bold; color: #007bff;"><?= $dias_presenca ?> dias</span>.
            </p>
            <button type="button" class="show-criterios-btn" id="toggleCriterios">
                Ver/Ocultar Critérios Detalhados de Pontuação
            </button>
            </div>

        <?= $mensagem ?>

        <div class="criterios-detalhados" id="criteriosDetalhes">
            <h2>📑 Critérios Detalhados para Avaliação (Referência)</h2>
            <?php foreach ($detalhes_avaliacao as $seccao_titulo => $itens_seccao): ?>
                <?php foreach ($itens_seccao as $codigo => $detalhe): ?>
                    <h3><?= htmlspecialchars($detalhe['nome']) ?> (Peso: <?= $detalhe['peso'] ?>)</h3>
                    <table>
                        <thead>
                            <tr>
                                <th style="width: 10%;">Pontos</th>
                                <th>Descrição do Critério</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            ksort($detalhe['criterios']);
                            foreach ($detalhe['criterios'] as $ponto => $descricao): 
                            ?>
                                <tr>
                                    <td style="text-align: center; font-weight: bold;"><?= $ponto ?></td>
                                    <td><?= htmlspecialchars($descricao) ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endforeach; ?>
            <?php endforeach; ?>
        </div>
        <form method="post" action="">
            <table class="evaluation-table">
                <thead>
                    <tr>
                        <th style="width: 5%;">Peso</th>
                        <th style="width: 35%;">Item de Avaliação</th>
                        <th style="width: 40%;">Critério Aplicado</th>
                        <th style="width: 10%;">Pontos Atribuídos (5, 10, 15, 20)</th>
                        <th style="width: 10%;">Pontuação Ponderada (Peso x Pontos)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    foreach ($detalhes_avaliacao as $seccao_titulo => $itens_seccao): 
                        foreach ($itens_seccao as $codigo => $detalhe): 
                            $item_data = $dados_avaliacao_salvos[$codigo];
                            $class = $item_data['automatico'] ? 'automatico' : '';
                            $ponto_atribuido = $item_data['ponto_atribuido'];
                            $ponto_sugerido = $item_data['pontuacao_sugerida']; 

                    ?>
                        <tr class="item-row <?= $class ?>" id="row_<?= $codigo ?>">
                            <td style="text-align: center;"><?= $detalhe['peso'] ?></td>
                            <td>
                                <?= htmlspecialchars($detalhe['nome']) ?>
                                <?php if ($item_data['automatico']): ?>
                                    <span class="sugestao-auto">(Sugestão: <?= $ponto_sugerido ?> pts)</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="criterio-text" data-criterios='<?= json_encode($detalhe['criterios']) ?>'>
                                    <?php if ($ponto_atribuido > 0 && isset($item_data['criterio_descricao'])): ?>
                                        **[<?= $ponto_atribuido ?> pontos]:** <?= htmlspecialchars($item_data['criterio_descricao']) ?>
                                    <?php else: ?>
                                         Selecione uma pontuação para ver o critério.
                                    <?php endif; ?>
                                </span>
                            </td>
                            <td style="text-align: center;">
                                <select name="pontuacao_atribuida[<?= $codigo ?>]" class="form-select" required data-item-codigo="<?= $codigo ?>">
                                    <option value="0" <?= ($ponto_atribuido == 0) ? 'selected' : '' ?>>--</option>
                                    <?php foreach ($pontuacoes_validas as $ponto): ?>
                                        <option value="<?= $ponto ?>" <?= ($ponto == $ponto_atribuido) ? 'selected' : '' ?>>
                                            <?= $ponto ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                            <td style="text-align: center;">
                                <span id="ponderada_<?= $codigo ?>"><?= number_format($item_data['pontuacao_ponderada'], 2, '.', '') ?></span>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <div class="result-box">
                <p>TOTAL PONDERADO ATUAL: <span style="font-size: 1.5em; color: #155724;"><?= number_format($total_final_ponderado, 2, '.', '') ?></span></p>
            </div>
            
            <h4 style="margin-top: 20px; border-bottom: 1px solid #ccc; padding-bottom: 5px;">Apreciação Geral do Avaliador:</h4>
            <textarea name="apreciacao_avaliador" placeholder="Digite aqui os comentários e apreciações sobre o desempenho do funcionário.">
                <?php 
                $stmt_apreciacao = $pdo->prepare("SELECT apreciacao_avaliador FROM avaliacao_final WHERE usuario_id = ? AND ano = ? AND trimestre = ?");
                $stmt_apreciacao->execute([$id_selecionado, $ANO_ATUAL, $trimestre_url]);
                $apreciacao_salva = $stmt_apreciacao->fetchColumn();
                echo htmlspecialchars($apreciacao_salva);
                ?>
            </textarea>

            <div class="button-group">
                <a href="documento_oficial.php?id=<?= $id_selecionado ?>&trimestre=<?= $trimestre_url ?>&modo=trimestral" class="link-ficha" target="_blank">
                    Ver Ficha Oficial de Impressão
                </a>
                <button type="submit" name="salvar_avaliacao">Salvar e Concluir Avaliação</button>
            </div>
        </form>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.querySelector('form');
            const totalBox = document.querySelector('.result-box span');
            const toggleButton = document.getElementById('toggleCriterios');
            const criteriosDetalhes = document.getElementById('criteriosDetalhes');
            
            // Definição dos Pesos (usados para o cálculo)
            const pesos = { 'I_QA': 0.2, 'II_PA': 0.3, 'III_R': 0.1, 'IV_AP': 0.2, 'V_RH': 0.2 };
            const selectElements = document.querySelectorAll('.form-select');
            
            // Função para calcular o total ponderado
            function calcularTotalPonderado() {
                let total = 0;
                selectElements.forEach(select => {
                    const codigo = select.getAttribute('data-item-codigo');
                    const ponto = parseInt(select.value) || 0;
                    const peso = pesos[codigo] || 0;
                    total += ponto * peso;
                });
                totalBox.textContent = total.toFixed(2);
            }

            // Função para buscar e aplicar o critério de descrição (usada para todos os Itens)
            function atualizarLinha(codigo, ponto) {
                const row = document.getElementById(`row_${codigo}`);
                if (!row) return;

                const ponderadaSpan = document.getElementById(`ponderada_${codigo}`);
                const criterioSpan = row.querySelector('.criterio-text');
                
                // 1. Atualizar Pontuação Ponderada
                const peso = pesos[codigo] || 0;
                const pontuacaoPonderada = ponto * peso;
                if (ponderadaSpan) {
                    ponderadaSpan.textContent = pontuacaoPonderada.toFixed(2);
                }

                // 2. Atualizar Descrição do Critério
                if (criterioSpan) {
                    const criteriosData = JSON.parse(criterioSpan.getAttribute('data-criterios'));
                    const descricao = criteriosData[ponto] || 'Selecione uma pontuação para ver o critério.';
                    
                    if (ponto > 0) {
                        criterioSpan.innerHTML = `**[${ponto} pontos]:** ${descricao}`;
                    } else {
                        criterioSpan.textContent = descricao;
                    }
                }
            }

            // Listener para todos os selects
            selectElements.forEach(select => {
                select.addEventListener('change', function() {
                    const codigo = this.getAttribute('data-item-codigo');
                    const ponto = parseInt(this.value) || 0;
                    
                    atualizarLinha(codigo, ponto);
                    calcularTotalPonderado();
                });
            });

            // Listener para o botão de alternância de critérios
            toggleButton.addEventListener('click', function() {
                if (criteriosDetalhes.style.display === 'none' || criteriosDetalhes.style.display === '') {
                    criteriosDetalhes.style.display = 'block';
                    this.textContent = 'Ocultar Critérios Detalhados de Pontuação';
                } else {
                    criteriosDetalhes.style.display = 'none';
                    this.textContent = 'Ver/Ocultar Critérios Detalhados de Pontuação';
                }
            });

            // Inicialização: Garante que os valores automáticos/salvos estão na interface
            selectElements.forEach(select => {
                const codigo = select.getAttribute('data-item-codigo');
                const ponto = parseInt(select.value) || 0;
                atualizarLinha(codigo, ponto); // Atualiza descrição e ponderada
            });
            calcularTotalPonderado(); // Garante que o total inicial está correto
        });
    </script>
</body>
</html>